const express = require("express");
const cors = require("cors");
const bodyParser = require("body-parser");
const mysql = require("mysql2/promise");
const path = require("path");
const multer = require("multer");
const { profile } = require("console");
const fs = require("fs");
const mime = require("mime-types"); // Add this line
const nodemailer = require("nodemailer");

const app = express();

app.use(bodyParser.json());
app.use(bodyParser.urlencoded({ extended: false }));
app.use(cors());
app.use(express.json());

// Serve static files from the 'images' directory
app.use("/images", express.static(path.join(__dirname, "images")));
app.use("/brochures", express.static(path.join(__dirname, "brochures")));

const pool = mysql.createPool({
  host: "localhost",
  user: "root",
  password: "",
  database: "chahira",
  waitForConnections: true,
  connectionLimit: 10,
  queueLimit: 0,
});

// Update the storage configuration for multer
const storage = multer.diskStorage({
  destination: (req, file, cb) => {
    // All images will be stored in the 'images' folder
    cb(null, path.join(__dirname, "images"));
  },
  filename: (req, file, cb) => {
    cb(null, Date.now() + "-" + file.originalname);
  },
});

const upload = multer({ storage: storage });
// Create a transporter using your working configuration

// brochures upload engine
const brochureStorage = multer.diskStorage({
  destination: (req, file, cb) => {
    if (file.fieldname === "file") {
      cb(null, path.join(__dirname, "brochures"));
    } else if (file.fieldname === "preview_image") {
      cb(null, path.join(__dirname, "images"));
    }
  },
  filename: (req, file, cb) => {
    cb(null, Date.now() + "-" + file.originalname);
  },
});

const uploadBrochure = multer({ storage: brochureStorage });

const transporter = nodemailer.createTransport({
  host: "smtp.gmail.com",
  port: 587,
  secure: false,
  auth: {
    user: "kelvinnyagah045@gmail.com",
    pass: "svov ptvu ihau rvjl",
  },
});

// brochures upload api endpoint
app.post(
  "/apid/brochures",
  uploadBrochure.fields([
    { name: "file", maxCount: 1 },
    { name: "preview_image", maxCount: 1 },
  ]),
  async (req, res) => {
    try {
      const { title } = req.body;
      const file_path = req.files["file"][0].filename;
      const preview_image = req.files["preview_image"][0].filename;

      const [result] = await pool.query(
        "INSERT INTO brochuresTable  (title, file_path, preview_image) VALUES (?, ?, ?)",
        [title, file_path, preview_image]
      );

      const newBrochure = {
        id: result.insertId,
        title,
        file_path: `${req.protocol}://${req.get(
          "host"
        )}/brochures/${file_path}`,
        preview_image: `${req.protocol}://${req.get(
          "host"
        )}/images/${preview_image}`,
      };

      res.status(201).json({
        message: "Brochure uploaded successfully",
        brochure: newBrochure,
      });
    } catch (error) {
      console.error(error);
      res.status(500).json({ message: "Server error" });
    }
  }
);
//api to delete a brochure
app.delete("/apid/brochures/:id", async (req, res) => {
  try {
    const { id } = req.params;

    // Get the brochure details before deleting
    const [brochures] = await pool.query(
      "SELECT file_path, preview_image FROM brochuresTable  WHERE id = ?",
      [id]
    );

    if (brochures.length === 0) {
      return res.status(404).json({ message: "Brochure not found" });
    }

    const brochure = brochures[0];

    // Delete the brochure file
    const brochurePath = path.join(__dirname, "brochures", brochure.file_path);
    try {
      await fs.promises.unlink(brochurePath);
      console.log(`Successfully deleted brochure file: ${brochurePath}`);
    } catch (err) {
      console.error(`Error deleting brochure file: ${brochurePath}`, err);
      // Continue execution even if file deletion fails
    }

    // Delete the preview image
    const imagePath = path.join(__dirname, "images", brochure.preview_image);
    try {
      await fs.promises.unlink(imagePath);
      console.log(`Successfully deleted preview image: ${imagePath}`);
    } catch (err) {
      console.error(`Error deleting preview image: ${imagePath}`, err);
      // Continue execution even if file deletion fails
    }

    // Delete the database record
    await pool.query("DELETE FROM brochuresTable  WHERE id = ?", [id]);

    res.json({ message: "Brochure deleted successfully" });
  } catch (error) {
    console.error("Error in delete brochure operation:", error);
    res.status(500).json({ message: "Server error", error: error.message });
  }
});
app.get("/api/:type/areas", async (req, res) => {
  const { type } = req.params;
  const tableName = tableMap[type];

  if (!tableName) {
    return res.status(400).json({ message: "Invalid project type" });
  }

  try {
    // Fetch all unique areas from the specified table
    const [rows] = await pool.query(
      `SELECT DISTINCT area FROM ${tableName} WHERE area IS NOT NULL AND area != '' ORDER BY area`
    );

    // Ensure we always return an array
    const areas = rows
      .map((row) => row.area || "")
      .filter((area) => area !== "");

    res.json(areas);
  } catch (error) {
    console.error("Error fetching areas:", error);
    // Return an empty array in case of error
    res.status(500).json([]);
  }
});

app.post("/api/send-email", async (req, res) => {
  const { name, email, phone, interest, message } = req.body;

  // Construct email text similar to your working version
  const emailText = `
    Name: ${name}
    Email: ${email}
    Phone: ${phone}
    Interest: ${interest}
    
    Message:
    ${message}
  `;

  const mailOptions = {
    from: email,
    to: "kelvinnyagah045@gmail.com", // Send to the user's email
    cc: "kelvinnyagah045@gmail.com", // Send a copy to yourself
    subject: `New Inquiry: ${interest}`,
    text: emailText, // Using plain text as in your working version
  };

  try {
    const info = await transporter.sendMail(mailOptions);
    console.log("Message sent: %s", info.messageId);
    res.status(200).json({
      message: "Email sent successfully",
      messageId: info.messageId,
    });
  } catch (error) {
    console.error("Error sending email:", error);
    res
      .status(500)
      .json({ message: "Failed to send email", error: error.message });
  }
});

// API endpoint to fetch all brochures
app.get("/apid/brochures", async (req, res) => {
  try {
    const [rows] = await pool.query(
      "SELECT * FROM brochuresTable  ORDER BY id DESC"
    );

    const brochuresData = rows.map((brochure) => ({
      ...brochure,
      preview_image: `${req.protocol}://${req.get("host")}/images/${
        brochure.preview_image
      }`,
      file_path: `${req.protocol}://${req.get("host")}/brochures/${
        brochure.file_path
      }`,
      mime_type: mime.lookup(brochure.file_path) || "application/octet-stream",
    }));

    res.json(brochuresData);
  } catch (error) {
    console.error(error);
    res.status(500).json({ message: "Server error" });
  }
});

// API endpoint to download a brochure
app.get("/apid/brochures/:id/download", async (req, res) => {
  try {
    const { id } = req.params;
    const [brochure] = await pool.query(
      "SELECT * FROM brochuresTable  WHERE id = ?",
      [id]
    );

    if (brochure.length === 0) {
      return res.status(404).json({ message: "Brochure not found" });
    }

    const filePath = path.join(__dirname, "brochures", brochure[0].file_path);
    const mimeType = mime.lookup(filePath) || "application/octet-stream";

    res.setHeader("Content-Type", mimeType);
    res.setHeader(
      "Content-Disposition",
      `attachment; filename="${brochure[0].title}"`
    );

    const fileStream = fs.createReadStream(filePath);
    fileStream.pipe(res);
  } catch (error) {
    console.error(error);
    res.status(500).json({ message: "Server error" });
  }
});

// API endpoint to update a brochure
app.put(
  "/apid/brochures/:id",
  upload.fields([
    { name: "file", maxCount: 1 },
    { name: "preview_image", maxCount: 1 },
  ]),
  async (req, res) => {
    try {
      const { id } = req.params;
      const { title } = req.body;
      let file_path = req.body.existing_file;
      let preview_image = req.body.existing_preview;

      if (req.files["file"]) {
        file_path = req.files["file"][0].filename;
        // Delete old file if it exists and is different from the new one
        if (req.body.existing_file && req.body.existing_file !== file_path) {
          try {
            await fs.unlink(
              path.join(__dirname, "brochures", req.body.existing_file)
            );
          } catch (err) {
            console.error("Error deleting old brochure file:", err);
          }
        }
      }

      if (req.files["preview_image"]) {
        preview_image = req.files["preview_image"][0].filename;
        // Delete old preview image if it exists and is different from the new one
        if (
          req.body.existing_preview &&
          req.body.existing_preview !== preview_image
        ) {
          try {
            await fs.unlink(
              path.join(__dirname, "images", req.body.existing_preview)
            );
          } catch (err) {
            console.error("Error deleting old preview image:", err);
          }
        }
      }

      await pool.query(
        "UPDATE brochuresTable  SET title = ?, file_path = ?, preview_image = ? WHERE id = ?",
        [title, file_path, preview_image, id]
      );

      const updatedBrochure = {
        id: parseInt(id),
        title,
        file_path: `${req.protocol}://${req.get(
          "host"
        )}/brochures/${file_path}`,
        preview_image: `${req.protocol}://${req.get(
          "host"
        )}/images/${preview_image}`,
      };

      res.json({
        message: "Brochure updated successfully",
        brochure: updatedBrochure,
      });
    } catch (error) {
      console.error(error);
      res.status(500).json({ message: "Server error" });
    }
  }
);

// Helper function to convert relative image paths to full URLs
const getFullImageUrls = (req, imageString) => {
  return imageString
    .split(",")
    .map((img) => `${req.protocol}://${req.get("host")}/images/${img.trim()}`);
};

// API endpoint to fetch landing page carousel data
app.get("/apid/landingpagecarousel", async (req, res) => {
  try {
    const [rows] = await pool.query(
      "SELECT * FROM landingpagecarousel ORDER BY id ASC"
    );

    const carouselData = rows.map((item) => ({
      ...item,
      images: getFullImageUrls(req, item.image),
    }));

    res.json(carouselData);
  } catch (error) {
    console.error(error);
    res.status(500).json({ message: "Server error" });
  }
});

// API endpoint to add new landing page carousel item
app.post(
  "/apid/landingpagecarousel",
  upload.array("images"),
  async (req, res) => {
    try {
      const { description, filename } = req.body;
      const images = req.files.map((file) => file.filename).join(",");

      const [result] = await pool.query(
        "INSERT INTO landingpagecarousel (image, description, filename) VALUES (?, ?, ?)",
        [images, description, filename]
      );

      const newItem = {
        id: result.insertId,
        images: getFullImageUrls(req, images),
        description,
        filename,
      };

      res.status(201).json({
        message: "Carousel item added successfully",
        item: newItem,
      });
    } catch (error) {
      console.error(error);
      res.status(500).json({ message: "Server error" });
    }
  }
);

// API endpoint to update a landing page carousel item
app.put(
  "/apid/landingpagecarousel/:id",
  upload.array("images"),
  async (req, res) => {
    try {
      const { id } = req.params;
      const { description, filename, existingImages } = req.body;
      let images = existingImages ? existingImages.split(",") : [];

      // Fetch the current item data
      const [currentItem] = await pool.query(
        "SELECT image FROM landingpagecarousel WHERE id = ?",
        [id]
      );

      if (currentItem.length === 0) {
        return res.status(404).json({ message: "Item not found" });
      }

      const currentImages = currentItem[0].image.split(",");

      // Handle image deletions
      const imagesToDelete = currentImages.filter(
        (img) => !images.includes(img)
      );

      for (const img of imagesToDelete) {
        await fs.promises.unlink(path.join(__dirname, "images", img.trim()));
      }

      // Handle new image uploads
      if (req.files && req.files.length > 0) {
        const newImages = req.files.map((file) => file.filename);
        images = [...images, ...newImages];
      }

      // Update the database
      await pool.query(
        "UPDATE landingpagecarousel SET image = ?, description = ?, filename = ? WHERE id = ?",
        [images.join(","), description, filename, id]
      );

      const updatedItem = {
        id,
        images: getFullImageUrls(req, images.join(",")),
        description,
        filename,
      };

      res.json({
        message: "Carousel item updated successfully",
        item: updatedItem,
      });
    } catch (error) {
      console.error(error);
      res.status(500).json({ message: "Server error" });
    }
  }
);

// API endpoint to delete a landing page carousel item
app.delete("/apid/landingpagecarousel/:id", async (req, res) => {
  try {
    const { id } = req.params;

    // Get the image filenames before deleting the record
    const [item] = await pool.query(
      "SELECT image FROM landingpagecarousel WHERE id = ?",
      [id]
    );

    if (item.length === 0) {
      return res.status(404).json({ message: "Item not found" });
    }

    // Delete the image files
    const images = item[0].image.split(",");
    for (const img of images) {
      try {
        await fs.promises.unlink(path.join(__dirname, "images", img.trim()));
      } catch (error) {
        console.error(`Error deleting image ${img}:`, error);
      }
    }

    // Delete the database record
    await pool.query("DELETE FROM landingpagecarousel WHERE id = ?", [id]);

    res.json({ message: "Carousel item deleted successfully" });
  } catch (error) {
    console.error(error);
    res.status(500).json({ message: "Server error" });
  }
});

// Dynamic endpoint that handles all project types
app.get("/api/:type/:id", async (req, res) => {
  const { type, id } = req.params;

  // Map frontend route names to database table names
  const tableMap = {
    designs: "designs",
    commercial: "commercial",
    renders: "renders",
    interiorDesign: "interiordesigns",
    public: "public",
    hospitality: "hospitality",
    masterPlanning: "masterplanning",
    mixedUse: "mixeduse",
    newProjects: "newprojects",
    singleResidential: "singleresidential",
    multiResidential: "multiresidential",
    allProjects: "allprojects",
  };

  const tableName = tableMap[type];

  if (!tableName) {
    return res.status(400).json({ message: "Invalid project type" });
  }

  try {
    const [rows] = await pool.query(`SELECT * FROM ${tableName} WHERE id = ?`, [
      id,
    ]);

    if (rows.length > 0) {
      // Convert relative image paths to full URLs
      const item = rows[0];
      item.images = getFullImageUrls(req, item.images);

      res.json(item);
    } else {
      res.status(404).json({ message: "Item not found" });
    }
  } catch (error) {
    console.error(error);
    res.status(500).json({ message: "Server error" });
  }
});

// Map frontend route names to database table names
const tableMap = {
  designs: "designs",
  commercial: "commercial",
  renders: "renders",
  interiorDesign: "interiordesigns",
  public: "public",
  hospitality: "hospitality",
  masterPlanning: "masterplanning",
  mixedUse: "mixeduse",
  newProjects: "newprojects",
  singleResidential: "singleresidential",
  multiResidential: "multiresidential",
  allProjects: "allprojects",
};

// Universal API endpoint to fetch all items for a specific type
app.get("/api/:type", async (req, res) => {
  const { type } = req.params;

  if (type === "allProjects") {
    try {
      let allItems = [];
      for (const [key, tableName] of Object.entries(tableMap)) {
        if (key !== "allProjects") {
          const [rows] = await pool.query(
            `SELECT * FROM ${tableName} ORDER BY id DESC`
          );
          const items = rows.map((item) => ({
            ...item,
            projectType: key,
            backgroundImage: getFullImageUrls(
              req,
              item.images.split(",")[0]
            )[0],
            images: getFullImageUrls(req, item.images),
          }));
          allItems = allItems.concat(items);
        }
      }
      res.json(allItems);
    } catch (error) {
      console.error(error);
      res.status(500).json({ message: "Server error" });
    }
  } else {
    const tableName = tableMap[type];

    if (!tableName) {
      return res.status(400).json({ message: "Invalid project type" });
    }

    try {
      const [rows] = await pool.query(
        `SELECT * FROM ${tableName} ORDER BY id DESC`
      );

      const items = rows.map((item) => ({
        ...item,
        backgroundImage: getFullImageUrls(req, item.images.split(",")[0])[0],
        images: getFullImageUrls(req, item.images),
      }));

      res.json(items);
    } catch (error) {
      console.error(error);
      res.status(500).json({ message: "Server error" });
    }
  }
});

// API endpoint to fetch profile data
app.get("/apid/profile", async (req, res) => {
  try {
    const [rows] = await pool.query("SELECT * FROM profile LIMIT 1");

    if (rows.length > 0) {
      const profile = rows[0];
      profile.profile_image = `${req.protocol}://${req.get("host")}/images/${
        profile.profile_image
      }`;
      res.json(profile);
    } else {
      res.status(404).json({ message: "Profile not found" });
    }
  } catch (error) {
    console.error(error);
    res.status(500).json({ message: "Server error" });
  }
});

// API endpoint to update profile data
app.put("/apid/profile", upload.single("profile_image"), async (req, res) => {
  try {
    const { full_name, header_text, about_text, quote } = req.body;
    let profile_image = req.body.existing_image;

    if (req.file) {
      profile_image = req.file.filename;
      // Delete old image file if it exists and is different from the new one
      if (
        req.body.existing_image &&
        req.body.existing_image !== profile_image
      ) {
        try {
          await fs.unlink(
            path.join(__dirname, "images", req.body.existing_image)
          );
        } catch (err) {
          console.error("Error deleting old profile image:", err);
        }
      }
    }

    await pool.query(
      "UPDATE profile SET full_name = ?, profile_image = ?, header_text = ?, about_text = ?, quote = ? WHERE id = 1",
      [full_name, profile_image, header_text, about_text, quote]
    );

    const updatedProfile = {
      full_name,
      profile_image: `${req.protocol}://${req.get(
        "host"
      )}/images/${profile_image}`,
      header_text,
      about_text,
      quote,
    };

    res.json({
      message: "Profile updated successfully",
      profile: updatedProfile,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({ message: "Server error" });
  }
});

// API endpoint to fetch all awards
app.get("/apid/awards", async (req, res) => {
  try {
    const [awards] = await pool.query(
      "SELECT * FROM awardsTable ORDER BY year DESC"
    );
    res.json(awards);
  } catch (error) {
    console.error(error);
    res.status(500).json({ message: "Server error" });
  }
});

// API endpoint to add a new award
app.post("/apid/awards", async (req, res) => {
  try {
    const { year, title, location } = req.body;
    const [result] = await pool.query(
      "INSERT INTO awardsTable (year, title, location) VALUES (?, ?, ?)",
      [year, title, location]
    );
    const [newAward] = await pool.query(
      "SELECT * FROM awardsTable WHERE id = ?",
      [result.insertId]
    );
    res.status(201).json({
      message: "Award added successfully",
      award: newAward[0],
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({ message: "Server error" });
  }
});

// API endpoint to update an award
app.put("/apid/awards/:id", async (req, res) => {
  try {
    const { id } = req.params;
    const { year, title, location } = req.body;

    const [result] = await pool.query(
      "UPDATE awardsTable SET year = ?, title = ?, location = ?, updated_at = CURRENT_TIMESTAMP WHERE id = ?",
      [year, title, location, id]
    );

    if (result.affectedRows === 0) {
      return res.status(404).json({ message: "Award not found" });
    }

    const [updatedAward] = await pool.query(
      "SELECT * FROM awardsTable WHERE id = ?",
      [id]
    );

    res.json({
      message: "Award updated successfully",
      award: updatedAward[0],
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({ message: "Server error" });
  }
});

// API endpoint to delete an award
app.delete("/apid/awards/:id", async (req, res) => {
  try {
    const { id } = req.params;

    const [result] = await pool.query("DELETE FROM awardsTable WHERE id = ?", [
      id,
    ]);

    if (result.affectedRows === 0) {
      return res.status(404).json({ message: "Award not found" });
    }

    res.json({ message: "Award deleted successfully" });
  } catch (error) {
    console.error(error);
    res.status(500).json({ message: "Server error" });
  }
});

// Update the API endpoint to add a new project
app.post("/api/:type", upload.array("images"), async (req, res) => {
  try {
    const { type } = req.params;
    const tableName = tableMap[type];

    if (!tableName) {
      return res.status(400).json({ message: "Invalid project type" });
    }

    const {
      name,
      area,
      description,
      status,
      site_area,
      project_area,
      project_type,
      project_team,
    } = req.body;

    // Images are now saved directly to the 'images' folder
    const images = req.files.map((file) => file.filename).join(",");

    const [result] = await pool.query(
      `INSERT INTO ${tableName} (name, area, description, status, site_area, project_area, project_type, project_team, images) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)`,
      [
        name,
        area,
        description,
        status,
        site_area,
        project_area,
        project_type,
        project_team,
        images,
      ]
    );

    const newProject = {
      id: result.insertId,
      name,
      area,
      description,
      status,
      site_area,
      project_area,
      project_type,
      project_team,
      images: getFullImageUrls(req, images),
    };

    res.status(201).json({
      message: "Project added successfully",
      project: newProject,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({ message: "Server error" });
  }
});
// API endpoint to update a project
app.put("/api/:type/:id", upload.array("images"), async (req, res) => {
  try {
    const { type, id } = req.params;
    const tableName = tableMap[type];

    if (!tableName) {
      return res.status(400).json({ message: "Invalid project type" });
    }

    const {
      name,
      area,
      description,
      status,
      site_area,
      project_area,
      project_type,
      project_team,
      existingImages,
    } = req.body;

    let images = existingImages ? existingImages.split(",") : [];

    if (req.files && req.files.length > 0) {
      const newImages = req.files.map((file) => file.filename);
      images = [...images, ...newImages];
    }

    const [oldProject] = await pool.query(
      `SELECT images FROM ${tableName} WHERE id = ?`,
      [id]
    );

    if (oldProject[0] && oldProject[0].images) {
      const oldImages = oldProject[0].images.split(",");
      const imagesToDelete = oldImages.filter(
        (img) => !images.includes(img) && !existingImages.includes(img)
      );

      // Delete old image files
      for (const img of imagesToDelete) {
        try {
          await fs.unlink(path.join(__dirname, "images", img));
        } catch (err) {
          console.error("Error deleting old image:", err);
        }
      }
    }

    const [result] = await pool.query(
      `UPDATE ${tableName} SET name = ?, area = ?, description = ?, status = ?, site_area = ?, project_area = ?, project_type = ?, project_team = ?, images = ? WHERE id = ?`,
      [
        name,
        area,
        description,
        status,
        site_area,
        project_area,
        project_type,
        project_team,
        images.join(","),
        id,
      ]
    );

    if (result.affectedRows === 0) {
      return res.status(404).json({ message: "Project not found" });
    }

    const updatedProject = {
      id,
      name,
      area,
      description,
      status,
      site_area,
      project_area,
      project_type,
      project_team,
      images: getFullImageUrls(req, images.join(",")),
    };

    res.json({
      message: "Project updated successfully",
      project: updatedProject,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({ message: "Server error" });
  }
});

// API endpoint to delete a project
app.delete("/api/:type/:id", async (req, res) => {
  try {
    const { type, id } = req.params;
    const tableName = tableMap[type];

    if (!tableName) {
      return res.status(400).json({ message: "Invalid project type" });
    }

    // Get the project details before deleting
    const [projects] = await pool.query(
      `SELECT images FROM ${tableName} WHERE id = ?`,
      [id]
    );

    if (projects.length === 0) {
      return res.status(404).json({ message: "Project not found" });
    }

    const project = projects[0];

    // Delete the project images
    if (project.images) {
      const images = project.images.split(",");
      for (const img of images) {
        const imagePath = path.join(__dirname, "images", img.trim());
        try {
          await fs.promises.unlink(imagePath);
          console.log(`Successfully deleted image: ${imagePath}`);
        } catch (err) {
          console.error(`Error deleting image: ${imagePath}`, err);
          // Continue execution even if file deletion fails
        }
      }
    }

    // Delete the database record
    const [result] = await pool.query(`DELETE FROM ${tableName} WHERE id = ?`, [
      id,
    ]);

    if (result.affectedRows === 0) {
      return res.status(404).json({ message: "Project not found" });
    }

    res.json({ message: "Project deleted successfully" });
  } catch (error) {
    console.error("Error in delete project operation:", error);
    res.status(500).json({ message: "Server error", error: error.message });
  }
});

// API endpoint to fetch contact information
app.get("/apid/contact-info", async (req, res) => {
  try {
    const [rows] = await pool.query(
      "SELECT * FROM contact_information  LIMIT 1"
    );

    if (rows.length > 0) {
      const contactInfo = rows[0];
      contactInfo.image = contactInfo.image
        ? `${req.protocol}://${req.get("host")}/images/${contactInfo.image}`
        : null;
      res.json(contactInfo);
    } else {
      res.status(404).json({ message: "Contact information not found" });
    }
  } catch (error) {
    console.error(error);
    res.status(500).json({ message: "Server error" });
  }
});

// API endpoint to update contact information
app.put("/apid/contact-info", upload.single("image"), async (req, res) => {
  try {
    const {
      address,
      phone,
      whatsapp,
      tiktok,
      pinterest,
      twitter,
      instagram,
      email,
      existing_image,
    } = req.body;

    let image = existing_image;

    if (req.file) {
      image = req.file.filename;
      // Delete old image file if it exists and is different from the new one
      if (existing_image && existing_image !== image) {
        try {
          await fs.unlink(path.join(__dirname, "images", existing_image));
        } catch (err) {
          console.error("Error deleting old contact image:", err);
        }
      }
    }

    const [result] = await pool.query(
      `UPDATE contact_information  SET 
      address = ?, phone = ?, whatsapp = ?, tiktok = ?, pinterest = ?, 
      twitter = ?, instagram = ?, email = ?, image = ?
      WHERE id = 1`,
      [
        address,
        phone,
        whatsapp,
        tiktok,
        pinterest,
        twitter,
        instagram,
        email,
        image,
      ]
    );

    if (result.affectedRows === 0) {
      // If no rows were updated, insert a new record
      await pool.query(
        `INSERT INTO contact_information  
        (address, phone, whatsapp, tiktok, pinterest, twitter, instagram, email, image) 
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)`,
        [
          address,
          phone,
          whatsapp,
          tiktok,
          pinterest,
          twitter,
          instagram,
          email,
          image,
        ]
      );
    }

    const updatedContactInfo = {
      address,
      phone,
      whatsapp,
      tiktok,
      pinterest,
      twitter,
      instagram,
      email,
      image: image
        ? `${req.protocol}://${req.get("host")}/images/${image}`
        : null,
    };

    res.json({
      message: "Contact information updated successfully",
      contactInfo: updatedContactInfo,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({ message: "Server error" });
  }
});

app.listen(3000, () => console.log("Server running on port 3000"));
